<?php if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package direct-project-innovation-initiative
* @subpackage controllers
* @filesource
*//** */

require_once 'restricted_controller.php';


/**
* @package direct-project-innovation-initiative
* @subpackage controllers
*/
class AdminPanel extends Restricted_controller {

	function __construct() {
		parent::__construct();
		$this->verify_permission("adminpanel"); 
	} 


	public function index()
	{	
		if($this->has_permission("reports")){
			redirect("adminpanel/reports");
		}
		else if($this->has_permission("logs")){
			redirect("adminpanel/logs");
		}
		else if($this->has_permission("manage_users")){
			redirect("adminpanel/manage_users");
		}
		else if($this->has_permission("manage_groups")){
			redirect("adminpanel/manage_groups");
		}
		else if($this->has_permission("contact_list")){
			redirect("adminpanel/contact_list");
		}
		else if($this->has_permission("distribution_lists")){
			redirect("adminpanel/distribution_lists");
		}
		else if($this->has_permission("themes")){
			redirect("adminpanel/themes");
		}
		else{
			redirect("auth");
		}
	}
	

	/*load log view */
	function logs($action = NULL, $value = NULL) {
		$this->verify_permission("logs");
		if(is_null($action)) { 
			redirect("adminpanel/logs/logins");
		}
		else {
			$data = $this->log_view($action,$value);
			$this->load->view("adminpanel/logs",$data);
		}
	}
	
	/*This function displays the export form view, meant for use with modal window or pop-up */
	public function export_form($log) {
		$this->load->view('adminpanel/export_form',array('log'=>$log));
	}
	
	/* This function exports a log for a given time period to csv format
	 */
	public function export($log) {
		$this->verify_permission("logs");
		$this->load->library('locale');
		$timezone_abbrev = $this->locale->timezone_abbr_from_name(date_default_timezone_get());
		
		$start_date = $this->input->post("start_date",TRUE);
		$end_date = $this->input->post("end_date",TRUE);
		$start_udate = strtotime($start_date);
		$end_udate = strtotime($end_date) ? strtotime($end_date) + 86400 : strtotime($end_date);
		$this->load->model('webmailmodel');
		$log_query = $this->webmailmodel->get_log_export($log,$start_udate,$end_udate);
		
		if($log == "logins") {
			$this->verify_permission("logs_logins");
			if($log_query) {
                header('Content-Type: application/download'); 
				header("Content-Transfer-Encoding: binary");
				header('Content-Disposition: attachment; filename="logins.csv"');
				echo "FOR OFFICIAL USE ONLY\n";
				echo "Log Date";
				if(isset($timezone_abbrev)) { echo ' (' . $timezone_abbrev . ')'; }
				echo ",Username,IP Address,Login Success,Error Message\n";
				for($i = 0; $i < $log_query->num_rows(); $i++) {
					$row = $log_query->row_array($i);
					foreach($row as $key => $val) { $row[$key] = str_replace("\"","\"\"",$val); }
					if($row["success"]) { $success = "Success"; } else { $success = "Failure"; }
					echo date("m/d/y h:i:s A",$row["login_time"]) . ",\"" . $row["username"] . "\"," . $row["ip_address"] . "," . $success . ",\"" . $row["error_msg"] . "\"\n";
				}
			}
		}
		else if($log == "sent_mail") {
			$this->verify_permission("logs_sent_mail");
			if($log_query) {
				header('Content-Type: application/download'); 
				header("Content-Transfer-Encoding: binary");
                header('Content-Disposition: attachment; filename="sent_mail.csv"');
				echo "FOR OFFICIAL USE ONLY\n";
				echo "Log Date";
				if(isset($timezone_abbrev)) { echo ' (' . $timezone_abbrev . ')'; }
				echo ",Sender,Recipient(s),Message Size (KB),Send Success\n";
				for($i = 0; $i < $log_query->num_rows(); $i++) {
					$row = $log_query->row_array($i);
					if($row["success"]) { $success = "Success"; } else { $success = "Failure"; }
					echo date("m/d/y h:i:s A",$row["time"]) . ",\"" . $this->sanatize_export($row["sender"]) . "\",\"" . implode(", ",$this->sanitize_export($this->json->decode($row["recipient"]))) . "\"," . $row["size"] . "," . $success . "\n";
				}
			}
		}
		else if($log == "received_mail") {
			$this->verify_permission("logs_received_mail");
			if($log_query) {
				header('Content-Type: application/download'); 
				header("Content-Transfer-Encoding: binary");
                header('Content-Disposition: attachment; filename="received_mail.csv"');
				echo "FOR OFFICIAL USE ONLY\n";
				echo "Log Date";
				if(isset($timezone_abbrev)) { echo ' (' . $timezone_abbrev . ')'; }
				echo ",Sender,Recipient(s),Message Size (KB),Receipt Success\n";
				for($i = 0; $i < $log_query->num_rows(); $i++) {
					$row = $log_query->row_array($i);
					if($row["success"]) { $success = "Success"; } else { $success = "Failure"; }
					echo date("m/d/y h:i:s A",$row["time"]) . ",\"" . $this->sanitize_export($row["sender"]) . "\",\"" . implode(", ",$this->sanitize_export($this->json->decode($row["recipient"]))) . "\"," . $row["size"] . "," . $success . "\n";
				}
			}
		}
		else if($log == "edit") {
			$this->verify_permission("logs_edit");
			if($log_query) {
				header('Content-Type: application/download'); 
				header("Content-Transfer-Encoding: binary");
                header('Content-Disposition: attachment; filename="edit_log.csv"');
				echo "FOR OFFICIAL USE ONLY\n";
				echo "Log Date";
				if(isset($timezone_abbrev)) { echo ' (' . $timezone_abbrev . ')'; }
				echo ",Target,Actor,Action\n";
				for($i = 0; $i < $log_query->num_rows(); $i++) {
					$row = $log_query->row_array($i);
					$target_id_query = $this->db->query("SELECT user_name FROM users WHERE user_id=" . $this->db->escape($row["target_user_id"]));
					$actor_id_query = $this->db->query("SELECT user_name FROM users WHERE user_id=" . $this->db->escape($row["actor_user_id"]));
					if($target_id_query && $actor_id_query) {
						$target_id_row = $target_id_query->row_array();
						$actor_id_row = $actor_id_query->row_array();
						$target_name = isset($target_id_row ["user_name"]) ? $target_id_row ["user_name"] : "Unknown User";
						$actor_name = isset($actor_id_row ["user_name"]) ? $actor_id_row ["user_name"] : "Unknown User";
						echo date("m/d/y h:i:s A",$row["edit_datetime"]) . ",\"" . $this->sanitize_export($target_name) . "\",\"" . $this->sanitize_export($actor_name) . "\"," . $this->sanitize_export($row["edit_action"]) . "\n";
					}
				}
			}
		}
		else if($log == "feedback") {
			$this->verify_permission("logs_feedback");
			if($log_query) {
				header('Content-Type: application/download'); 
				header("Content-Transfer-Encoding: binary");
                header('Content-Disposition: attachment; filename="feedback.csv"');
				echo "FOR OFFICIAL USE ONLY\n";
				echo "Log Date";
				if(isset($timezone_abbrev)) { echo ' (' . $timezone_abbrev . ')'; }
				echo ",Username,Feedback Type,Feedback Comments\n";
				for($i = 0; $i < $log_query->num_rows(); $i++) {
					$row = $log_query->row_array($i);
					$actor_id_query = $this->db->query("SELECT user_name FROM users WHERE user_id=" . $this->db->escape($row["user_id"]));
					if($actor_id_query) {
						$actor_id_row = $actor_id_query->row_array();
						$actor_name = isset($actor_id_row["user_name"]) ? $actor_id_row["user_name"] : "Unknown User";
						echo date("m/d/y h:i:s A",$row["feedback_datetime"]) . ",\"" . $this->sanitize_export($actor_name) . "\",\"" . $this->sanitize_export($row["feedback_type"]) . "\"," . $this->sanitize_export($row["feedback_comments"]) . "\n";
					}
				}
			}
		}
		else { show_404(); }
	}
	/* This function loads statistics from the database into html tables which are then converted into
	 * chart visualizations by a jquery library 
	 */
	public function chart($chart = NULL) {
		$data["title"] = PORTAL_TITLE_PREFIX . "Admin Panel";
		if(is_null($chart)) { $chart = "sent_messages"; }
		switch ($chart) {
			case "sent_messages":
					$this->verify_permission("reports_sent_messages");
					//create total messages sent per day chart
					$stmt = $this->db->query("SELECT time AS date, COUNT(time) AS msg_count FROM mail_log WHERE inbound_outbound=0 AND time BETWEEN (datediff(ss, '19700101', GetUtcDate()) - 518400) AND (datediff(ss, '19700101', GetUtcDate())) GROUP BY time");
					$i = 0;
					$chartdata = array();
					$results = array();
					foreach($stmt->result_array() as $row) {
						$date = date("m/d", $row['date']);
						$count = $row['msg_count'];
						if(array_key_exists($date,$results)) { $results[$date] = $results[$date] + $count; }
						else { $results[$date] = $count; }
					}
					foreach($results as $key => $row) {
						$chartdata[$i]["date"] = $key;
						$chartdata[$i]["count"] = $row;
						$i++;
					}
					//add dates with no records to chart
					$j = 0;
				for($i = (date("U") - (86400*6)); $i <= date("U"); $i = $i + 86400) {
					if(isset($chartdata[$j]) && $chartdata[$j]["date"] != date("m/d",$i)) { array_splice($chartdata,$j,0,array(array("date" => date("m/d",$i), "count" => 0))); }
					else if(!isset($chartdata[$j])){ $chartdata[$j]["date"] = date("m/d",$i); $chartdata[$j]["count"] = 0; }
					$j++;
				}
				$data['chart_title'] = 'Messages Sent Daily Chart';
				$data["chart"] = $this->markup_for_jquery_chart("line","Messages Sent Daily",$chartdata, "sent_daily_chart","650px","325px");
				$data['hidden_chart'] = '<table><caption>Table representing data in Messages Sent Daily chart</caption><tr><th>Date</th><th>Messages Sent</th></tr>';
				foreach($chartdata as $cdata) {
					$data['hidden_chart'] .= '<tr><td>' . $cdata['date'] . '</td><td>' . $cdata['count'] . '</td></tr>';
				}
				$data['hidden_chart'] .= '</table>';
				break;
			case "message_success":
				$this->verify_permission("reports_message_success");
				//create success vs fail sent messages chart
				$stmt = $this->db->query("SELECT SUM(CASE WHEN success = 1 THEN 1 ELSE 0 END) AS success, SUM(CASE WHEN success = 0 THEN 1 ELSE 0 END) AS fail FROM mail_log WHERE inbound_outbound=0");
				$chartdata = array();
				foreach($stmt->result_array() as $row) {
					$chartdata["success"]["label"] = "Succeeded";
					$chartdata["success"]["val"] = $row['success'];
					$chartdata["fail"]["label"] = "Failed"; 
					$chartdata["fail"]["val"] = $row['fail']; 
				}
				$data['chart_title'] = 'Send Success/Failure Chart';
				$data["chart"] = $this->markup_for_jquery_chart("pie","Send Success/Failure",$chartdata, "success_fail_chart", "650px","325px","['#0162A2','#FFA500']");
				$data['hidden_chart'] = '<table>';
				$data['hidden_chart'] .= '<caption>Table representing data in Send Success/Failure chart</caption>';
				$data['hidden_chart'] .= '<tr><th scope="col">Data Type</th><th scope="col">' . $chartdata['success']['label'] . '</th><th scope="col">' . $chartdata['fail']['label'] . '</th></tr>'; 
				$data['hidden_chart'] .= '<tr><th scope="row">Count</th><td>' . $chartdata['success']['val'] . '</td><td>' . $chartdata['fail']['val'] . '</td></tr>';
				$data['hidden_chart'] .= '<tr><th scope="row">Percentage of Total</th><td>' . round((($chartdata['success']['val']/($chartdata['success']['val']+$chartdata['fail']['val']))*100),2).'%'.'</td><td>' . round((($chartdata['fail']['val']/($chartdata['success']['val']+$chartdata['fail']['val']))*100),2).'%'. '</td></tr>';
				$data['hidden_chart'] .= '</table>';
				break;
			case "average_size":
				$this->verify_permission("reports_average_size");
				//create average message size per day chart
				$stmt = $this->db->query("SELECT time AS date, AVG(size) AS msg_size FROM mail_log WHERE inbound_outbound=0 AND time BETWEEN (datediff(ss, '19700101', GetUtcDate()) - 518400) AND (datediff(ss, '19700101', GetUtcDate())) GROUP BY time");
				$chartdata = array();
				$results = array();
				foreach($stmt->result_array() as $row) {
						$date = date("m/d", $row['date']);
						$size = $row['msg_size'];
						if(array_key_exists($date,$results)) { 
							$results[$date]['size'] = $results[$date]['size'] + $size; 
							$results[$date]['count'] = $results[$date]['count'] + 1;
						}
						else { $results[$date]['size'] = $size; $results[$date]['count'] = 1; }
				}
				$i = 0;
				foreach($results as $key => $row) {
					$chartdata[$i]["date"] = $key;
					$chartdata[$i]["size"] = round(($row['size'] / $row['count']),2);
					$i++;
				}
				//add dates with no records to chart
				$j = 0;
				for($i = (date("U") - (86400*6)); $i <= date("U"); $i = $i + 86400) {
					if(isset($chartdata[$j]) && $chartdata[$j]["date"] != date("m/d",$i)) { array_splice($chartdata,$j,0,array(array("date" => date("m/d",$i), "size" => 0))); }
					else if(!isset($chartdata[$j])){ $chartdata[$j]["date"] = date("m/d",$i); $chartdata[$j]["size"] = 0; }
					$j++;
				}
				$data['chart_title'] = 'Avg. Daily Message Size (KB) Chart';
				$data["chart"] = $this->markup_for_jquery_chart("line","Avg. Daily Message Size (KB)",$chartdata, "avg_size_chart","650px","325px");
				$data['hidden_chart'] = '<table><caption>Table representing data in Average Daily Message Size chart</caption><tr><th>Date</th><th>Size (KB)</th></tr>';
				foreach($chartdata as $cdata) {
					$data['hidden_chart'] .= '<tr><td>' . $cdata['date'] . '</td><td>' . $cdata['size'] . '</td></tr>';
				}
				$data['hidden_chart'] .= '</table>';
				break;
			case "unique_logins":
				$this->verify_permission("reports_unique_logins");
				//create unique logins per day chart
				$stmt = $this->db->query("SELECT login_time AS date, username, ip_address FROM logins WHERE success=1 AND (login_time BETWEEN (datediff(ss, '19700101', GetUtcDate()) - 518400) AND (datediff(ss, '19700101', GetUtcDate())))");
				$chartdata = array();
				$results = array();
				foreach($stmt->result_array() as $row) {
						$date = date("m/d", $row['date']);
						$username = $row['username'];
						$ip = $row['ip_address'];
						$user_ip = $username . '#' . $ip;
						if(array_key_exists($date,$results)) {
							if(array_key_exists($user_ip,$results[$date])) {
								$results[$date][$user_ip]['count'] = $results[$date][$user_ip]['count'] + 1;
							}
							else { $results[$date][$user_ip]['count'] = 1; }
						}
						else { 
							$results[$date][$user_ip]['count'] = 1;
						}
				}
				foreach($results as $date => $logins) {
					$day_count = 0;
					foreach($logins as $user_ip) {
						$day_count++;
					}
					$results[$date]['count'] = $day_count;
				}
				$i = 0;
				foreach($results as $key => $row) {
					$chartdata[$i]["date"] = $key;
					$chartdata[$i]["count"] = $row['count'];
					$i++;
				}
				
				//add dates with no records to chart
				$j = 0;
				for($i = (date("U") - (86400*6)); $i <= date("U"); $i = $i + 86400) {
					if(isset($chartdata[$j]) && $chartdata[$j]["date"] != date("m/d",$i)) { array_splice($chartdata,$j,0,array(array("date" => date("m/d",$i), "count" => 0))); }
					else if(!isset($chartdata[$j])){ $chartdata[$j]["date"] = date("m/d",$i); $chartdata[$j]["count"] = 0; }
					$j++;
				}
				$data['chart_title'] = 'Unique Successful Logins Daily Chart';
				$data["chart"] = $this->markup_for_jquery_chart("bar","Unique Successful Logins Daily",$chartdata, "unique_login_chart","650px","325px","['blue']");
				$data['hidden_chart'] = '<table><caption>Table representing data in Unique Successful Logins Daily chart</caption><tr><th>Date</th><th>Login Count</th></tr>';
				foreach($chartdata as $cdata) {
					$data['hidden_chart'] .= '<tr><td>' . $cdata['date'] . '</td><td>' . $cdata['count'] . '</td></tr>';
				}
				$data['hidden_chart'] .= '</table>';
				break;
			case "login_success":
				$this->verify_permission("reports_login_success");
				//create success vs fail login chart
				$stmt = $this->db->query("SELECT SUM(CASE WHEN success = 1 THEN 1 ELSE 0 END) AS success, SUM(CASE WHEN success = 0 THEN 1 ELSE 0 END) AS fail FROM logins");
				$chartdata = array();
				foreach($stmt->result_array() as $row) {
					$chartdata["success"]["label"] = "Succeeded";
					$chartdata["success"]["val"] = $row['success'];
					$chartdata["fail"]["label"] = "Failed"; 
					$chartdata["fail"]["val"] = $row['fail']; 
				}
				$data['chart_title'] = 'Login Success/Failure Chart';
				$data["chart"] = $this->markup_for_jquery_chart("pie","Login Success/Failure",$chartdata, "login_success_fail_chart", "650px","325px","['#0162A2','#FFA500']");
				$data['hidden_chart'] = '<table>';
				$data['hidden_chart'] .= '<caption>Table representing data in Login Success/Failure chart</caption>';
				$data['hidden_chart'] .= '<tr><th scope="col">Data Type</th><th scope="col">' . $chartdata['success']['label'] . '</th><th scope="col">' . $chartdata['fail']['label'] . '</th></tr>'; 
				$data['hidden_chart'] .= '<tr><th scope="row">Count</th><td>' . $chartdata['success']['val'] . '</td><td>' . $chartdata['fail']['val'] . '</td></tr>';
				$data['hidden_chart'] .= '<tr><th scope="row">Percentage of Total</th><td>' . round((($chartdata['success']['val']/($chartdata['success']['val']+$chartdata['fail']['val']))*100),2).'%'.'</td><td>' . round((($chartdata['fail']['val']/($chartdata['success']['val']+$chartdata['fail']['val']))*100),2).'%'. '</td></tr>';
				$data['hidden_chart'] .= '</table>';
				break;
			case "login_errors":
				$this->verify_permission("reports_login_errors");
				//create failed login error chart
				$stmt = $this->db->query("SELECT TOP(5) username, error_msg FROM logins WHERE success = 0 ORDER BY login_time DESC");
				$chartdata = array();
				$i = 0;
				foreach($stmt->result_array() as $row) {
					$chartdata[$i]["username"] = $row['username'];
					$chartdata[$i]["error"] = $row['error_msg'];
					$i++;
				}
				$data['chart_title'] = 'Recent Login Failure Reasons Chart';
				$data["chart"] = $this->markup_for_jquery_chart("table","Recent Login Failure Reasons",$chartdata, "failed_login_chart", "650px","325px","",FALSE);
				$data['hidden_chart'] = '';
				break;
			case "attachment_types":
				$this->verify_permission("reports_attachment_types");
				//create unique attachment types chart
				$stmt = $this->db->query("SELECT attachment_types FROM mail_log");
				$chartdata = array();
				foreach($stmt->result_array() as $row) {
					if(strlen(trim($row["attachment_types"])) > 0) { 
						$counts = array_count_values(explode(" ",trim($row["attachment_types"])));
						foreach($counts as $key => $count) { 
							if(!array_key_exists($key,$chartdata)) { 
								$chartdata[$key]["type"] = $key;
								$chartdata[$key]["count"] = $count; 
							}	 
							else { 
								$chartdata[$key]["type"] = $key;
								$chartdata[$key]["count"] = $chartdata[$key]["count"] + $count;
							}	
						}
					}
					else if(strlen(trim($row["attachment_types"])) == 0) {
						if(!array_key_exists("none",$chartdata)) { 
							$chartdata["none"]["type"] = "none";
							$chartdata["none"]["count"] = 0;
						}
						else { 
							$chartdata["none"]["type"] = "none";
							$chartdata["none"]["count"] = $chartdata["none"]["count"] + 1;
						}
					}
				}
				$data['chart_title'] = 'Unique Attachment Type Volumes Chart';
				$data["chart"] = $this->markup_for_jquery_chart("bar","Unique Attachment Type Volumes",$chartdata,"unique_attachments_chart","650px","325px","['blue']");
				$data['hidden_chart'] = '<table><caption>Table representing data in Unique Attachment Type Volumes chart</caption><tr><th>Type</th><th>Attachment Count</th></tr>';
				foreach($chartdata as $cdata) {
					$data['hidden_chart'] .= '<tr><td>' . $cdata['type'] . '</td><td>' . $cdata['count'] . '</td></tr>';
				}
				$data['hidden_chart'] .= '</table>';
				break;
			default:
				$data["chart"] = "Unrecognized chart type";
				break;
		}
		//load view
		$this->load->view("adminpanel/chart",$data);
	}
	
	public function reports() {	
		$this->verify_permission("reports");
		$data["title"] = PORTAL_TITLE_PREFIX . "Admin Panel";
		$this->load->view("adminpanel/reports",$data);
	}
		
	public function archive_setting($action = NULL) {
		$this->verify_permission("archive_settings");
		if(is_null($action)) {
	
			$data["title"] = PORTAL_TITLE_PREFIX . "Archive Settings";
			$message_archive_settings = $this->db->query("SELECT TOP 1 * FROM dbo.archive_settings")->result();
			$data['message_archive_settings'] = $message_archive_settings;
			
			$this->load->view("adminpanel/archive_settings",$data);
		}
		
	}
	
	public function themes($action = NULL) {
		$this->verify_permission("themes");
		if(is_null($action)) {

			$data["title"] = PORTAL_TITLE_PREFIX . "Admin Panel";
			$get_themes = $this->db->query("SELECT * FROM themes");
			$data["themes"] = array();
			if($get_themes) {
				for($i = 0; $i < $get_themes->num_rows(); $i++) {
					array_push($data["themes"],$get_themes->row_array($i));
				}
			}
			$this->load->view("adminpanel/themes",$data);
		}
		//TO-DO: These else statements on failure should lead to error message given to user
		else if($action == "load_theme") {
			$theme_id = $this->input->post("theme_id",TRUE);
			$check_theme =  $this->db->query("SELECT theme_id FROM themes WHERE theme_id = " . $this->db->escape($theme_id));
			if($check_theme->num_rows() > 0) {
				$update_default_theme = $this->db->query("UPDATE themes SET theme_is_default=0 WHERE theme_id !=" . $this->db->escape($theme_id));
				if($update_default_theme) {
					$update_default_theme = $this->db->query("UPDATE themes SET theme_is_default=1 WHERE theme_id=" . $this->db->escape($theme_id));
					if($update_default_theme) { 
						$update_theme = $this->db->query("UPDATE users SET user_theme=" . $this->db->escape($theme_id));
						if($update_theme) {
							redirect("adminpanel/themes");
						}
						else { redirect("adminpanel/themes"); }
					}
					else { redirect("adminpanel/themes"); }
				}
			}
			else { redirect("adminpanel/themes"); }
		}
		else { show_404(); }
	}	
	
	/*This function creates a view with which to view and edit users
	 */
	public function manage_users($action = NULL, $value = NULL) {
		$this->verify_permission("manage_users");
		$data["title"] = PORTAL_TITLE_PREFIX . "Admin Panel";
		if(is_null($action)) { $action = "page"; }
		if($action == "page"  || $action == "removed") {
			$removed = (($action == "removed")?1:0);
			$user_got = false;
			$limit = "";
			if($this->has_permission("manage_users_all")){
				$user_got = true;
			}else if($this->has_permission("manage_users_facilities")){
				$resource = '/direct/account/in_facility/mailbox/'.$this->session->userdata('username');
				$resource .= '/format/json';
				$response = @$this->api_model->webservice_call($resource,'GET');
				if($response->http_status === 200){
					$users = $response->response->accounts;
					
					for ($count = 0; $count < count($users); $count++){
						$users[$count] = $this->db->escape($users[$count]);
					}
					$limit = " and user_name in (".implode(",", $users).") ";
					$user_got = true;
				}
			}
			if(is_null($value)) { $data['page_start'] = 0; $data['page'] = 1; }
			else { $data['page'] = $value; $data['page_start'] = (($data['page']-1)*ADMINPANEL_DISPLAY_PER_PAGE);  }
			$get_user_count = $this->db->query("SELECT COUNT(user_name) AS count FROM users WHERE user_deleted_flag=".$removed." ".$limit." AND user_is_group=0");
			
			if($get_user_count && $user_got) {
				$user_count_arr = $get_user_count->row_array();
				$data['user_count'] = $user_count_arr["count"];
				$get_users = $this->db->query("SELECT user_name FROM (SELECT ROW_NUMBER() OVER (ORDER BY user_name) AS RowNum, user_name,user_deleted_flag,user_is_group FROM users WHERE user_deleted_flag =".$removed." AND user_is_group=0 ".$limit.") AS rows WHERE user_deleted_flag =".$removed." ".$limit." AND user_is_group=0 AND RowNum > " . ($data['page_start']) . " AND RowNum <= " . ($data['page_start'] + ADMINPANEL_DISPLAY_PER_PAGE) . " ORDER BY RowNum");
				if($get_users && $get_users->num_rows()) {
					for($i = 0; $i < $get_users->num_rows(); $i++) {
						$row = $get_users->row_array($i);
						$properties = array("uid","givenname","sn","initials","o","departmentnumber","telephonenumber","mobile","title","physicaldeliveryofficename");
						$filter = "(&(ObjectClass=person)(&(uid=" . $row["user_name"] . ")))";
						if($removed){
							$ldap_result = $this->ldap->search(NULL,NULL,$properties,$filter,LDAP_DELETED_ACCOUNTS_DN);
						}
						else{	
							$ldap_result = $this->ldap->search(NULL,NULL,$properties,$filter);
						}
						$data['users'][$i] = $ldap_result[0];
					}
					//($data['users'] Sort User result table by Last name and Name
					foreach ($data['users'] as $key => $rowUser) {
						$LastName[$key]  = $rowUser['sn'];
						$GivenName[$key] = $rowUser['givenname'];
					}
					array_multisort($LastName, SORT_ASC, $GivenName, SORT_ASC, $data['users']);
				}
			}
			//load view
			if($data['user_count']){
				array_walk_recursive($data['users'], function (&$value) {
					$value = htmlentities($value);
				});
			}
			$this->load->view("adminpanel/manage_users/".($removed?"removed":"index"),$data);
		}
		else if($action == "user") {
			if(is_null($value)) { show_404(); }
			$data['username'] = $value;
			$data['roles'] = $this->role_model->get_roles();
			$this->can_edit_user($value);
			//check that user value corresponds to valid user
			$get_user = $this->db->query("SELECT user_deleted_flag,user_id,user_name,user_mail,user_ext_notify_flag,user_ext_group_notify_flag  FROM users WHERE user_name=" . $this->db->escape($data['username']));
			if($get_user) {
				if($get_user->num_rows() == 1) { 
					$row = $get_user->row_array();
					if($row["user_deleted_flag"] == "0"){
						$dn = LDAP_ACCOUNTS_DN;
					}else{
						$dn = LDAP_DELETED_ACCOUNTS_DN;
					}
					$resource = '/direct/account/mailbox/'.$row["user_name"];
					$resource .= '/format/json';
					$response = @$this->api_model->webservice_call($resource,'GET');
					if($response->http_status === 200){
						$data = array_merge($data, get_object_vars($response->response));
						$data['management'] = true;
					}else {
						$data['management'] = false;
					}
					$properties = array('employeeType','dn');
					$filter = "(&(ObjectClass=person)(&(uid=" . $row["user_name"] . ")))";
					$ldap_result = $this->ldap->search(NULL,NULL,$properties,$filter,$dn);
					$data['selected_roles'] = $this->role_model->get_roles_membership($ldap_result[0]["dn"]);
					if(count($ldap_result) == 1) {
						if(isset($ldap_result[0]['employeetype'])) { 
							if($ldap_result[0]['employeetype'] == 'mailboxhidden') { $data['show_personal'] = FALSE; }
							else { $data['show_personal'] = TRUE; }
						}
						else { $data['show_personal'] = TRUE; }
						$group_search = $this->ldap->get_admin_group_membership($ldap_result[0]["dn"]);
						$groups = array();
						for($i = 0; $i < $group_search["count"]; $i++) {
							array_push($groups,$group_search[$i]["dn"]);
						}
						if(in_array(LDAP_ADMIN_GROUP1,$groups)) {
							$data["adminlevel"] = 1;
						}
						else { $data["adminlevel"] = 0; }
					}
					if(isset($row["user_ext_notify_flag"])) { if($row["user_ext_notify_flag"] == "1") { $data['notify'] = TRUE; } else { $data['notify'] = FALSE; } } else { $data['notify'] = FALSE; }
					if(isset($row['user_ext_group_notify_flag'])) { if($row['user_ext_group_notify_flag'] == '1') { $data['group_notify'] = TRUE; } else { $data['group_notify'] = FALSE; } } else { $data['group_notify'] = FALSE; }
					
					if($row["user_deleted_flag"] == "0"){
						$this->load->view("adminpanel/manage_users/edit_user",$data);
					}
					else{
						$this->load->view("adminpanel/manage_users/edit_removed_user",$data);
					} 
				}
				else{
					redirect("/adminpanel/manage_users/");
				}
			}
			else {
				redirect("/adminpanel/manage_users/");
			}
		}
		else if($action == "process_request") {
			if(array_key_exists("update",$this->input->post(NULL,TRUE))) { $this->update_user_account(); }
            else if(array_key_exists("update_removed",$this->input->post(NULL,TRUE))) { $this->update_user_account(FALSE); }
			else if(array_key_exists("remove",$this->input->post(NULL,TRUE))) { $this->remove_user(); }
			else if(array_key_exists("restore",$this->input->post(NULL,TRUE))) { $this->restore_user(); }
			else if(array_key_exists("create",$this->input->post(NULL,TRUE))) { 
				if($this->create_user()) { redirect("/adminpanel/manage_users/"); } 
				else { 
					$this->session->set_flashdata('form_data',$this->input->post(NULL,TRUE));
					redirect("/adminpanel/manage_users/create"); 
				}
			}
		}
		else if($action == "create") {
			$this->verify_permission("manage_users_create");
		 	$this->load->view("adminpanel/manage_users/account_creation",$data);
		}
		else { show_404(); }
	}
	
	//This function loads the main index page for the contact list. 
	public function contact_list() {
		//HAVE TO ADD THIS TO PERMISSIONS
		$this->verify_permission('manage_users');
		$data["title"] = PORTAL_TITLE_PREFIX . 'Admin Panel';
		$contacts = $this->db->query('SELECT * FROM admin_contact_list');
		$result_arr = array();
		for($i = 0; $i < $contacts->num_rows(); $i++) {
			$row = $contacts->row_array($i);
			$displayname = $row['last_name'] . ', ' . $row['first_name'];
			if(isset($row['middle_name']) && (strlen($row['middle_name']) > 0)) { $displayname .= ' ' . $row['middle_name']; }
			$contact_arr = array('type'=>'contact', 'displayname' => $displayname, 'address' => $row['direct_address'], 'first_name' => $row['first_name'], 'last_name' => $row['last_name'], 'middle_name' => $row['middle_name'], 'organization' => $row['organization'], 'department' => $row['department'], 'sharing' => $row['sharing'], 'job_title' => $row['title'], 'telephone' => $row['telephone'], 'contact_id' => $row['contact_id']);
			//don't return blank/null attributes
			foreach($contact_arr as $attr => $val) {
				if(!isset($val) || strlen($val) <= 0) {
					unset($contact_arr[$attr]);
				}
			}
			array_push($result_arr,$contact_arr);
		}
		
		//sort result array (natural order, case insensitive)
		usort($result_arr, function( $el1, $el2) { return strnatcasecmp( $el1['displayname'], $el2['displayname']); });
		
		array_walk_recursive($result_arr, function (&$value) {
			$value = htmlentities($value);
		});
		$data['initial_json'] = $this->json->encode($result_arr);
		$this->load->view('adminpanel/contact_list',$data);
	}
	
	//This function loads the page where users can create a new contact for the contact list.
	public function admin_contact_create() {
		$this->verify_permission('manage_users');
		$data["title"] = PORTAL_TITLE_PREFIX . 'Admin Panel';
		$this->load->view('adminpanel/contact_create',$data);
	}
	
	//This function loads the fancybox where a user can edit a contact from the contact list and change the contact's sharing settings.
	public function edit_contact($contact_id = null) {
		$this->verify_permission('manage_users');
		$data['title'] = PORTAL_TITLE_PREFIX . 'Admin Panel';
		$contact = $this->db->query('SELECT * FROM admin_contact_list WHERE contact_id=' . $this->db->escape($contact_id))->row_array(0);
		if(empty($contact)){
			show_404();
		}
		$shared_with = array_flip(array_filter(explode(',', $contact['sharing'])));
		$group_not_shared_with = array(); 
		$properties = array('displayname','mail','uid');
		$person_result = $this->ldap->search(null,null,$properties);
		$not_shared_with = array();
		foreach($person_result as $user) {
			$mailbox = Mailbox::find_one( array('user_name' => $user['uid']) );
			//only add actual users, this prevents issues with junk in LDAP that is out of sync
			if(Mailbox::is_an_entity($mailbox)) {
				$user['is_group'] = ($mailbox->is_group()) ? TRUE : FALSE; 
				if(array_key_exists($user['uid'], $shared_with)) {
					$user_info = array($user['uid'], $user['displayname']);
					$shared_with[$user['uid']] = $user_info;
				}
				else if(!array_key_exists($user['uid'], $shared_with) && $user['is_group']) { $group_not_shared_with[] = $user; }
				else { $not_shared_with[] = $user; }
			}
		}
		$data['contact_json'] = $this->json->encode($contact); //stores all of the contact's info
		$data['user_json'] = $this->json->encode($not_shared_with); //stores the username, displayname and mail address of all the direct users not currently sharing this contact.
		$data['group_json'] = $this->json->encode($group_not_shared_with); //stores the username, displayname and mail address of all the direct users not currently sharing this contact.
		$data['sharing_json'] = $this->json->encode($shared_with); //stores the username, displayname and mail address of all the direct users currently sharing this contact.
		$this->load->view('adminpanel/edit_contact',$data);
	}
	
	//This function removes a user from the list of users a contact is shared with.
	public function remove_contact_share() {
		$uid = $this->input->post('uid', TRUE);
		$contact_id = $this->input->post('contact_id', TRUE);
		$contact_id = trim($contact_id, '"');
		$uid = trim($uid, '"');
		$this->verify_permission('manage_users');
		$query = $this->db->query('UPDATE admin_contact_list SET sharing = REPLACE(sharing, ' . $this->db->escape($uid.',') . ', \'\') WHERE contact_id = '. $this->db->escape($contact_id));
		if($query) { }
		else { }
		redirect('/adminpanel/edit_contact/' . $contact_id);
	}
	
	//This function adds a user to the list of users a contact is shared with.
	public function add_contact_share() {
		$this->verify_permission('manage_users');
		$users = $this->input->post('users_select', TRUE);
		$groups = $this->input->post('groups_select', TRUE);
		$contact_id = $this->input->post('contact_id', TRUE);
		
		if(is_array($users) && is_array($groups)) { $uids = array_merge($users, $groups); }
		else if(empty($users)) { $uids = $groups; }
		else if(empty($groups)) { $uids = $users; }
		
		foreach($uids as $uid) {
			$mailbox = Mailbox::find_one( array('user_name' => $uid) );
			if(Mailbox::is_an_entity($mailbox)) {
				$query = $this->db->query('UPDATE admin_contact_list SET sharing = sharing + ' . $this->db->escape($uid.',') . ' WHERE contact_id='. $this->db->escape($contact_id));
			}
		}
		if($query) {  }
		else {  }
		redirect('/adminpanel/edit_contact/' . $contact_id);
	}
	
	/* Alias to edit_admin_contact to create new contacts */
#TODO - couldn't this just be accomplished with a route?	
	public function create_admin_contact() {
		$this->verify_permission('manage_users');
		$this->edit_admin_contact();
	}
	
	/** This function is used to both edit and create contacts for the contact list.
	* If an address is specified it is being called from the editContact fancybox page and is used to edit an already existing contact.
	* If no address is specified it is being called from the contact_create page and is creating a new contact.
	* The two are combined into one function to save space so the input capturing doesn't have to be written out twice.
	**/
	public function edit_admin_contact($contact_id = NULL) {
		$this->verify_permission('manage_users');
		
		//check if contact id provided exists
		if(isset($contact_id)) {
			$contact = $this->db->query('SELECT * FROM admin_contact_list WHERE contact_id=' . $this->db->escape($contact_id))->row_array(0);
			if(empty($contact)) { return FALSE; }
		}
		
		$this->form_validation->set_rules('first_name','First Name','required','xss_clean');
		$this->form_validation->set_rules('last_name','Last Name','required','xss_clean');
		$this->form_validation->set_rules('dir_addr','Direct Address','required|valid_email|callback_valid_trusted_address','xss_clean');	
		if($this->form_validation->run() == true) {
			//gather post items into variables
			$first_post = $this->input->post('first_name',TRUE);
			$last_post = $this->input->post('last_name',TRUE);
			$dir_addr_post = $this->input->post('dir_addr',TRUE);
			$middle_post = $this->input->post('middle_name',TRUE);
			$organization_post = $this->input->post('organization',TRUE);
			$department_post = $this->input->post('department',TRUE);
			$telephone_post = $this->input->post('telephone',TRUE);
			$mobile_post = $this->input->post('mobile',TRUE);
			$title_post = $this->input->post('job_title',TRUE);
			//then do other required items
			if(!empty($first_post)) { $first = $first_post; } else { return FALSE; }
			if(!empty($last_post)) { $last = $last_post; } else { return FALSE; }
			if(!empty($dir_addr_post)) { $dir_addr = $dir_addr_post; } else { return FALSE; }
			//process optional items
			if(!empty($middle_post)) { $middle = $middle_post; } else { $middle = NULL;  }
			if(!empty($organization_post)) { $organization = $organization_post; } else { $organization = NULL;  }
			if(!empty($department_post)) { $department = $department_post; } else { $department = NULL;  }
			if(!empty($telephone_post)) { $telephone = $telephone_post; } else { $telephone = NULL;  }
			if(!empty($title_post)) { $title = $title_post; } else { $title = NULL;  }
			if(!empty($mobile_post)) { $mobile = $mobile_post; } else { $mobile = NULL;  }
			
			if (isset($contact_id)) {
				$query = $this->db->query('UPDATE admin_contact_list SET first_name ='. $this->db->escape($first) . ', middle_name=' . $this->db->escape($middle) . ', last_name='. $this->db->escape($last) . ', organization='. $this->db->escape($organization) . ', department=' . $this->db->escape($department) . ', telephone=' . $this->db->escape($telephone) . ', mobile=' . $this->db->escape($mobile) . ', title=' . $this->db->escape($title). ', direct_address=' . $this->db->escape($dir_addr) . ' WHERE contact_id=' . $this->db->escape($contact_id));
			}
			else {
				$query = $this->db->query('INSERT INTO admin_contact_list (first_name,middle_name,last_name,organization,department,telephone,mobile,title,direct_address,sharing) VALUES (' . $this->db->escape($first) . ',' . $this->db->escape($middle) . ',' . $this->db->escape($last) . ',' . $this->db->escape($organization) . ',' . $this->db->escape($department) . ',' . $this->db->escape($telephone) . ',' . $this->db->escape($mobile) . ',' . $this->db->escape($title) . ',' . $this->db->escape($dir_addr) . ',' . $this->db->escape(',') . ')');
				 $contact_id = $this->db->insert_id();
			}
			if($query) {
				$this->session->set_flashdata('message_class', 'success');
				$this->session->set_flashdata('message', 'Contact information saved successfully.');
			}
			else {
				$this->session->set_flashdata('message_class', 'error');
				$this->session->set_flashdata('message', 'Failed to save contact information.');
			}
		}
		else {
			//validation fails
			$form_data = $this->input->post(NULL,TRUE);	
			$this->session->set_flashdata('form_data', $form_data);
			if(is_array($form_data)) {
				foreach($form_data as $key => $input) {
					if(strlen(form_error($key)) > 0) { $validation_errors[$key] = form_error($key); }
				}
				$this->session->set_flashdata('validation_errors',$validation_errors);
			}
		}
		if(isset($contact_id)) {
			redirect('/adminpanel/edit_contact/' . $contact_id);
		}
		else {
			redirect('/adminpanel/admin_contact_create');
		}
	}
	//This function removes a contact from the contact list.
	public function remove_admin_contact() {
		$contact_id = $this->input->post('contact_id', TRUE);
		$this->load->database();
		$this->db->query("DELETE FROM admin_contact_list WHERE contact_id=". $this->db->escape($contact_id));
		redirect("/adminpanel/contact_list/");
	}
	
	//This function loads the webmail ldap users for the user to pick from when deciding who to share the contact with.
	public function contact_list_search($input=null) {
		$properties = array('displayname','mail','uid');
		$person_result = $this->ldap->search(rawurldecode($input),null,$properties);
		
		array_walk_recursive($person_result, function (&$value) {
			$value = htmlentities($value);
		});
		//sort result array (natural order, case insensitive)
		usort($person_result, function( $el1, $el2) { return strnatcasecmp( $el1['displayname'], $el2['displayname']); });
		
		echo $this->json->encode($person_result);
	}
	
	public function distribution_lists() {
		$this->paginate_distribution_lists(1);
	}
	
	public function paginate_distribution_lists($page) {
		$this->verify_permission("distribution_lists");
		$lists = $this->public_distribution_list_model->find(array());
		$data['list_count'] = count($lists);
		$data['page'] = $page;
		$data['lists'] = array_slice($lists,(($page-1)*ADMINPANEL_DISPLAY_PER_PAGE),ADMINPANEL_DISPLAY_PER_PAGE);
		//sort array by keys, using natural order (i.e. test1, test2, ... , test10 vs. test1, test10, test2)
		uksort($data['lists'], function($a, $b) { return strnatcasecmp($a,$b); });
		
		$data['title'] = PORTAL_TITLE_PREFIX . 'Admin Panel'; //set title of page
		$this->load->view('adminpanel/manage_lists/index',$data);
	}
	
	public function distribution_lists_removed() {
		$this->paginate_distribution_lists_removed(1);
	}
	
	public function paginate_distribution_lists_removed($page) {
		$this->verify_permission("distribution_lists");
		$lists = $this->public_distribution_list_model->find_deleted(array());
		$data['list_count'] = count($lists);
		$data['page'] = $page;
		$data['lists'] = array_slice($lists,(($page-1)*ADMINPANEL_DISPLAY_PER_PAGE),ADMINPANEL_DISPLAY_PER_PAGE);
		$data['title'] = PORTAL_TITLE_PREFIX . 'Admin Panel'; //set title of page
		$this->load->view('adminpanel/manage_lists/removed',$data);
	}
	
	//displays the edit form - for save, see edit_distribution_lists()
	public function distribution_lists_edit($id) { 
		$this->verify_permission("distribution_lists");
		$active_tab = '/adminpanel/distribution_lists/' ;
		$form = 'adminpanel/edit_distribution_list';
		$this->_distribution_lists_edit($id, 'public_distribution_list_model', compact('active_tab', 'form'));
	}
	
	//displays the edit form - for save, see distribution_lists_edit_removed()
	public function distribution_lists_edit_removed($id) {
		$this->verify_permission("distribution_lists");
		$active_tab = '/adminpanel/distribution_lists/removed' ;
		$form = 'adminpanel/edit_removed_distribution_list';
		$this->_distribution_lists_edit($id, 'deleted_public_distribution_list_model', compact('active_tab', 'form'));
	}
	
	/**
	* Displays & submits form to create public distribution lists.
	*/
	public function create_distribution_list() {
		$this->verify_permission("distribution_lists_create");
		$this->load->library('audit');
		$title = PORTAL_TITLE_PREFIX . 'Admin Panel';
		$form_data = array();
		$validation_errors = array();
		$error = '';
		
		if(!empty($_POST)){
			$list_display_name = $this->input->post('list_display_name',TRUE);
			$list_description = $this->input->post('list_description',TRUE);
				
			//set valdiation rules
			if($this->form_validation->run('distribution_list') === TRUE) {
				$attributes = array(
					'name' => $list_display_name,
					'cn' => $list_display_name,
					'description' => $list_description,
				);
				$id = $this->public_distribution_list_model->create($attributes);
				if($this->public_distribution_list_model->formatted_like_an_id($id)){
					$list = $this->public_distribution_list_model->find_one(array('cn'=> $list_display_name));
					$this->audit->log_event('edit',array(0,$this->user->id,'Created list: ' . $list_display_name. ' ('.$list['id'].')',date('U')));
					redirect('adminpanel/distribution_lists/list/'.$id);
				}else{
					$error = 'Something went wrong and your distribution list was not created.  Please try again in a moment. '.
							 'If you continue to see this error, please contact a site administrator.';
				}
			}
			else {
				//validation fails
				$form_data = $this->input->post(NULL,TRUE);
				$form_data['list_display_name']=html_entity_decode($form_data['list_display_name'],ENT_QUOTES);
				$form_data['list_description']=html_entity_decode($form_data['list_description'],ENT_QUOTES);
				if(is_array($form_data)) {
					foreach($form_data as $key => $input) {
						if(strlen(form_error($key)) > 0) { $validation_errors[$key] = form_error($key); }
					}
					$this->session->set_flashdata('validation_errors',$validation_errors);
				}
			}
		}
		$this->load->view('adminpanel/manage_lists/create',compact('error', 'form_data', 'title', 'validation_errors'));		
	}
		
	// saves the edit form - for the form setup, see distribution_lists_edit()
	public function edit_distribution_list() {
		return $this->_edit_distribution_list('public_distribution_list_model');
	}
	
	// saves the edit form - for the form setup, see distribution_lists_edit_removed()
	public function edit_removed_distribution_list() {
		return $this->_edit_distribution_list('deleted_public_distribution_list_model');
	}
	
	public function remove_distribution_list($list_id) {
		$this->verify_permission("distribution_lists");
		if($this->public_distribution_list_model->exists($list_id)) {
			$list = $this->public_distribution_list_model->find_one($list_id);
			$this->public_distribution_list_model->delete($list_id);
			$this->load->library('audit');
			if($list !== null){
				$this->audit->log_event('edit',array(0,$this->user->id,'Removed list: ' . $list['name']. ' ('.$list_id.')',date('U')));
			}
		}
		else { show_404(); }
		redirect('adminpanel/distribution_lists');
	}
	
	public function restore_distribution_list($list_id) {
		$this->verify_permission("distribution_lists");
		if($this->public_distribution_list_model->exists_deleted($list_id)) {
			$this->public_distribution_list_model->restore_deleted($list_id);
			$list = $this->public_distribution_list_model->find_one($list_id);
			$this->load->library('audit');
			if($list !== null){
				$this->audit->log_event('edit',array(0,$this->user->id,'Restored list: ' . $list['name']. ' ('.$list_id.')',date('U')));
			}
		}
		else { show_404(); }
		redirect('adminpanel/distribution_lists/removed');
	}
	
	/* This function displays the add user to group form view, meant for use with a modal window or pop-up */
	public function add_user_to_group_form($group,$active) {
		$this->verify_permission("manage_groups");
		$this->load->view('adminpanel/manage_groups/add_user_form',array('group'=>$group,'active'=>$active));
	}
	
	
	public function id_required($str, $field) {
		switch($field) {
			case 'piv':
				$this->form_validation->set_message('id_required', 'The %s field is required when the EDIPI field is not provided.');
				return (strlen(trim($str)) <= 0) ? FALSE : TRUE;
				break;
			case 'edipi':
				$this->form_validation->set_message('id_required', 'The %s field is required when the PIV ID field is not provided.');
				return (strlen(trim($str)) <= 0) ? FALSE : TRUE;
				break;
			default:
				return FALSE;
				break;
		}
	}
	
	public function update_archive_setting() {
		$this->form_validation->set_rules("archive_age","Message Age for Archive","required|is_natural_no_zero","xss_clean");
		$this->form_validation->set_rules("frequency","Frequency","required","xss_clean");
		$frequency = $this->input->post("frequency",TRUE);
		$archive_age = $this->input->post("archive_age",TRUE);
		$archive_age_unit = $this->input->post("archive_age_unit",TRUE);
		if($this->form_validation->run() == true) {
			$frequency = $this->input->post("frequency",TRUE);
			$archive_age = $this->input->post("archive_age",TRUE);
			$frequency_array = array('none', 'daily', 'weekly', 'biweekly', 'monthly', 'bimonthly', 'six_months', 'yearly');	
			$archive_age_unit_array = array('day', 'week', 'month', 'year');
				
			if(in_array($frequency, $frequency_array) && in_array($archive_age_unit, $archive_age_unit_array)) {
				$message_archive_setting_id = $this->db->query("SELECT TOP 1 id FROM dbo.archive_settings")->result();
				$message_archive_setting_id = $message_archive_setting_id[0]->id;
				$update = $this->db->query("UPDATE dbo.archive_settings SET frequency = " . $this->db->escape($frequency) . ", archive_age = " . $this->db->escape($archive_age) . ", archive_age_unit = " . $this->db->escape($archive_age_unit) . " WHERE id = " . $this->db->escape($message_archive_setting_id));
			}
			
			redirect('/adminpanel/archive_setting');
		}
		else {
			$form_data = $this->input->post(NULL,TRUE);
			$validation_errors = array();
			if(is_array($form_data)) {
				foreach($form_data as $key => $input) {
					if(strlen(form_error($key)) > 0) { $validation_errors[$key] = form_error($key); }
				}
				$this->session->set_flashdata('validation_errors',$validation_errors);
			}
			redirect('/adminpanel/archive_setting');
		}
	}
	
	public function ajax_group_member_search($input = NULL,$sizelimit = NULL, $echo = TRUE) {
		if(!is_null($sizelimit)) { $properties = NULL; }
		else {  $properties = array('displayname','objectclass','mail','uid','givenname','initials','sn','physicaldeliveryofficename','o','departmentnumber','mobile','telephonenumber','title');}
		
		//only allow echoing of JSON data if requested with AJAX
		if(IS_AJAX) {
			if($echo) { echo $this->json->encode($this->ldap->search(rawurldecode($input),$sizelimit,$properties,'(&(userPassword=*)(|(displayName='.$input.'*)(uid='.$input.'*)))')); }
		}
		//if not echoing, allow return
		if(!$echo) { return $this->json->encode($this->ldap->search(rawurldecode($input),$sizelimit,$properties,'(&(userPassword=*)(|(displayName='.$input.'*)(uid='.$input.'*)))')); }
	}
	
////////////////////////////////////////////////////////////////////
// PROTECTED METHODS
// Helper methods for this controller which are not web-accessible.
////////////////////////////////////////////////////////////////////

	//helper method to display the edit form for active & removed distros
	protected function _distribution_lists_edit($id, $model, $data) {
		if(!$this->is->nonempty_string($model)) return $this->error->should_be_a_nonempty_string($model);
				
		//first, check to make sure we have a valid list id
		$this->load->model($model,'list_model');
		if(!$this->list_model->formatted_like_an_id($id)) show_404();		
		$list = $this->list_model->find_one(array('id'=>$id));
		if(empty($list)) show_404();
		
		//load data for form
		$data['title'] = PORTAL_TITLE_PREFIX . 'Admin Panel';
		$data['list_id'] = $id;
		$data['list'] = $list;
		$data['addresses'] = $this->list_model->addresses_for_list($list);
		$data['form_data'] = array('list_display_name' => $list['name'],
								   'list_description' => $list['description']);

		//set up the options for the multiselect
		$user_options = array();
		foreach($this->get_user_search() as $user){
			$email = strtolower(trim($user['id']));
			$user_options[$email] =  $user['name'] . ' ('.$email.')';
		}
		
		$external_addresses = array_diff($data['addresses'], array_keys($user_options));
		foreach($external_addresses as $external_address){
			$user_options[$external_address] = $external_address;
		}
		natcasesort($user_options);		
		$data['user_options'] = $user_options;
		
		$this->load->view('adminpanel/manage_lists/edit_list', $data);
	}

	// helper method used by both methods that save edit forms for distros (edit_distribution_list && edit_removed_distribution_list) 
	protected function _edit_distribution_list($model) {
		if(!$this->is->nonempty_string($model)) return $this->error->should_be_a_nonempty_string($model);				
		$this->load->model($model,'list_model');
		
		$list_display_name = $this->input->post('list_display_name',TRUE);
		$list_description = $this->input->post('list_description',TRUE);
		$new_members = $this->input->post('list_members',TRUE);
		if(empty($new_members)) $new_members = array(); //when nothing's selected, we don't always get an array back
		$new_members = array_map('strtolower', $new_members);
		$list_id = $this->input->post('list_id',TRUE);
		
		//set valdiation rules
		if($this->form_validation->run('distribution_list') === TRUE) {
			//update display name and description
			$attributes = array(
				'cn' => $list_display_name,
				'description' => $list_description,
			);
			$this->list_model->update($list_id,$attributes);
			//update addresses
			$old_members = $this->list_model->addresses_for_list($list_id);
			$addresses_to_add = array_diff($new_members,$old_members);
			$addresses_to_remove = array_diff($old_members,$new_members);
			
			foreach($addresses_to_add as $address) {
				 $this->list_model->add_address_to_list($list_id,$address);
			}
			foreach($addresses_to_remove as $address) {
				 $this->list_model->remove_address_from_list($list_id,$address);
			}
		}
		else {
			//validation fails
			$form_data = $this->input->post(NULL,TRUE);
			$validation_errors = array();
			if(is_array($form_data)) {
				foreach($form_data as $key => $input) {
					if(strlen(form_error($key)) > 0) { $validation_errors[$key] = form_error($key); }
				}
				echo $this->json->encode($validation_errors);
			}
		}
	}
	
	protected function log_view($action = NULL, $value = NULL) {		
		$data["title"] = PORTAL_TITLE_PREFIX . "Admin Panel";
		$this->load->library('locale');
		$timezone_abbrev = $this->locale->timezone_abbr_from_name(date_default_timezone_get());
		if($action == "logins") {
			$this->verify_permission("logs_logins");
			$data["active_tab"] = "logins";
			if(is_null($value)) { $page_start = 0; $page = 1; }
			else { $page = $value; $page_start = (($page-1)*ADMINPANEL_LOG_DISPLAY_PER_PAGE);  }
			$logins = $this->db->query("SELECT * FROM (SELECT ROW_NUMBER() OVER (ORDER BY login_time DESC) AS RowNum, * FROM logins) AS rows WHERE RowNum > " . ($page_start) . " AND RowNum <= " . ($page_start + ADMINPANEL_LOG_DISPLAY_PER_PAGE) . " ORDER BY RowNum");
			$log_count_query = $this->db->query("SELECT COUNT(id) AS login_count FROM logins");
			if($logins && $log_count_query) {
				$row = $log_count_query->row_array();
				$log_count = $row["login_count"];
				$view_str = "<span style=\"float: right;\"><a href=\"javascript:exportCSV('logins');\">Export</a></span>";
				$view_str .= "<h2>System Logs &#8594; Login Logs</h2>";
				$view_str .= $this->generate_pagination($page,$log_count,ADMINPANEL_LOG_DISPLAY_PER_PAGE,'/adminpanel/logs/logins/');
				$view_str .= "<table class=\"log\">";
				$view_str .= '<caption class="hidden_context">Log of Login Records</caption>';
				$view_str .= "<tr><th>Log Date";
				if(isset($timezone_abbrev)) { $view_str .= ' (' . $timezone_abbrev .')'; }
				$view_str .= "</th><th>Username</th><th>IP Address</th><th>Login Success</th><th>Error Message</th></tr>";
				for($i = 0; $i < $logins->num_rows(); $i++) {
					$row = $logins->row_array($i);
					if($row["success"]) { $success = "Success"; } else { $success = "Failure"; }
					$view_str .= "<tr><td>" . htmlentities(date("m/d/y h:i:s A",$row["login_time"])) . "</td><td>" . htmlentities($row["username"]) . "</td><td>" . htmlentities($row["ip_address"]) . "</td><td>" . $success . "</td><td>" . htmlentities($row["error_msg"]) . "</td></tr>";
				}
				$view_str .= "</table>";
			}
		}
		else if($action == "sent_mail") {
			$this->verify_permission("logs_sent_mail");
			$data["active_tab"] = "sent_mail";
			if(is_null($value)) { $page_start = 0; $page = 1; }
			else { $page = $value; $page_start = (($page-1)*ADMINPANEL_LOG_DISPLAY_PER_PAGE);  }
			$sent_mail = $this->db->query("SELECT * FROM (SELECT ROW_NUMBER() OVER (ORDER BY time DESC) AS RowNum, * FROM mail_log WHERE inbound_outbound=0) AS rows WHERE inbound_outbound = 0 AND RowNum > " . ($page_start) . " AND RowNum <= " . ($page_start + ADMINPANEL_LOG_DISPLAY_PER_PAGE) . " ORDER BY RowNum");
			$log_count_query = $this->db->query("SELECT COUNT(id) AS sent_count FROM mail_log WHERE inbound_outbound=0");
			if($sent_mail && $log_count_query) {
				$row = $log_count_query->row_array();
				$log_count = $row["sent_count"];
				$view_str = "<span style=\"float: right;\"><a href=\"javascript:exportCSV('sent_mail');\">Export</a></span>";
				$view_str .= "<h2>System Logs &#8594; Sent Mail Logs</h2>";
				$view_str .= $this->generate_pagination($page,$log_count,ADMINPANEL_LOG_DISPLAY_PER_PAGE,'/adminpanel/logs/sent_mail/');
				$view_str .= "<table class=\"log\">";
				$view_str .= '<caption class="hidden_context">Log of Sent Mail</caption>';
				$view_str .= "<tr><th>Log Date";
				if(isset($timezone_abbrev)) { $view_str .= ' (' . $timezone_abbrev .')'; }
				$view_str .="</th><th>Sender</th><th>Recipient(s)</th><th>Message Size (KB)</th><th>Send Success</th></tr>";
				for($i = 0; $i < $sent_mail->num_rows(); $i++) {
					$row = $sent_mail->row_array($i);
					if($row["success"]) { $success = "Success"; } else { $success = "Failure"; }
					$view_str .= "<tr><td>" . htmlentities(date("m/d/y h:i:s A",$row["time"])) . "</td><td>" . htmlentities($row["sender"]) . "</td><td>" . htmlentities(implode(", ",$this->json->decode($row["recipient"]))) . "</td><td>" . $row["size"] . "</td><td>" . $success . "</td></tr>";
				}
				$view_str .= "</table>";
			}
		}
		else if($action == "received_mail") {
			$this->verify_permission("logs_received_mail");
			$data["active_tab"] = "received_mail";
			if(is_null($value)) { $page_start = 0; $page = 1; }
			else { $page = $value; $page_start = (($page-1)*ADMINPANEL_LOG_DISPLAY_PER_PAGE);  }
			$sent_mail = $this->db->query("SELECT * FROM (SELECT ROW_NUMBER() OVER (ORDER BY time DESC) AS RowNum, * FROM mail_log WHERE inbound_outbound=1) AS rows WHERE inbound_outbound = 1 AND RowNum > " . ($page_start) . " AND RowNum <= " . ($page_start + ADMINPANEL_LOG_DISPLAY_PER_PAGE) . " ORDER BY RowNum");
			$log_count_query = $this->db->query("SELECT COUNT(id) AS sent_count FROM mail_log WHERE inbound_outbound=1");
			if($sent_mail && $log_count_query) {
				$row = $log_count_query->row_array();
				$log_count = $row["sent_count"];
				$view_str = "<span style=\"float: right;\"><a href=\"javascript:exportCSV('received_mail');\">Export</a></span>";
				$view_str .= "<h2>System Logs &#8594; Received Mail Logs</h2>";
				$view_str .= $this->generate_pagination($page,$log_count,ADMINPANEL_LOG_DISPLAY_PER_PAGE,'/adminpanel/logs/received_mail/');
				$view_str .= "<table class=\"log\">";
				$view_str .= '<caption class="hidden_context">Log of Received Mail</caption>';
				$view_str .= "<tr><th>Log Date";
				if(isset($timezone_abbrev)) { $view_str .= ' (' . $timezone_abbrev .')'; }
				$view_str .="</th><th>Sender</th><th>Recipient(s)</th><th>Message Size (KB)</th><th>Receipt Success</th></tr>";
				for($i = 0; $i < $sent_mail->num_rows(); $i++) {
					$row = $sent_mail->row_array($i);
					if($row["success"]) { $success = "Success"; } else { $success = "Failure"; }
					$view_str .= "<tr><td>" . htmlentities(date("m/d/y h:i:s A",$row["time"])) . "</td><td>" . htmlentities($row["sender"]) . "</td><td>" . htmlentities(implode(", ",$this->json->decode($row["recipient"]))) . "</td><td>" . $row["size"] . "</td><td>" . $success . "</td></tr>";
				}
				$view_str .= "</table>";
			}
		}
		else if($action == "edit") {
			$this->verify_permission("logs_edit");
			$data["active_tab"] = "edit";
			if(is_null($value)) { $page_start = 0; $page = 1; }
			else { $page = $value; $page_start = (($page-1)*ADMINPANEL_LOG_DISPLAY_PER_PAGE);  }
			$edits = $this->db->query("SELECT * FROM (SELECT ROW_NUMBER() OVER (ORDER BY edit_datetime DESC) AS RowNum, * FROM edit_log) AS rows WHERE RowNum > " . ($page_start) . " AND RowNum <= " . ($page_start + ADMINPANEL_LOG_DISPLAY_PER_PAGE) . " ORDER BY RowNum");
			$log_count_query = $this->db->query("SELECT COUNT(edit_id) AS edit_count FROM edit_log");
			if($edits && $log_count_query) {
				$row = $log_count_query->row_array();
				$log_count = $row["edit_count"];
				$view_str = "<span style=\"float: right;\"><a href=\"javascript:exportCSV('edit');\">Export</a></span>";
				$view_str .= "<h2>System Logs &#8594; User Edit Logs</h2>";
				$view_str .= $this->generate_pagination($page,$log_count,ADMINPANEL_LOG_DISPLAY_PER_PAGE,'/adminpanel/logs/edit/');
				$view_str .= "<table class=\"log\">";
				$view_str .= '<caption class="hidden_context">Log of Edit Records</caption>';
				$view_str .= "<tr><th>Log Date";
				if(isset($timezone_abbrev)) { $view_str .= ' (' . $timezone_abbrev .')'; }
				$view_str .= "</th><th>Target</th><th>Actor</th><th>Action</th></tr>";
				for($i = 0; $i < $edits->num_rows(); $i++) {
					$row = $edits->row_array($i);
					$target_id_query = $this->db->query("SELECT user_name FROM users WHERE user_id=" . $this->db->escape($row["target_user_id"]));
					$actor_id_query = $this->db->query("SELECT user_name FROM users WHERE user_id=" . $this->db->escape($row["actor_user_id"]));
					if($target_id_query && $actor_id_query) {
						$target_id_row = $target_id_query->row_array();
						$actor_id_row = $actor_id_query->row_array();
						$target_name = isset($target_id_row ["user_name"]) ? $target_id_row ["user_name"] : "Unknown User";
						$actor_name = isset($actor_id_row ["user_name"]) ? $actor_id_row ["user_name"] : "Unknown User";
						$view_str .= "<tr><td>" . htmlentities(date("m/d/y h:i:s A",$row["edit_datetime"])) . "</td><td>" . htmlentities($target_name) . "</td><td>" . htmlentities($actor_name) . "</td><td>" . htmlentities($row["edit_action"]) . "</td></tr>";
					}
				}
				$view_str .= "</table>";
			}
		}
		else if($action == "feedback") {
			$this->verify_permission("logs_feedback");
			$data["active_tab"] = "feedback";
			if(is_null($value)) { $page_start = 0; $page = 1; }
			else { $page = $value; $page_start = (($page-1)*ADMINPANEL_LOG_DISPLAY_PER_PAGE);  }
			$feedback = $this->db->query("SELECT * FROM (SELECT ROW_NUMBER() OVER (ORDER BY feedback_id DESC) AS RowNum, * FROM feedback) AS rows WHERE RowNum > " . ($page_start) . " AND RowNum <= " . ($page_start + ADMINPANEL_LOG_DISPLAY_PER_PAGE) . " ORDER BY RowNum");
			$log_count_query = $this->db->query("SELECT COUNT(feedback_id) AS feedback_count FROM feedback");
			if($feedback && $log_count_query) {
				$row = $log_count_query->row_array();
				$log_count = $row["feedback_count"];
				$view_str = "<span style=\"float: right;\"><a href=\"javascript:exportCSV('feedback');\">Export</a></span>";
				$view_str .= "<h2>System Logs &#8594; User Feedback Logs</h2>";
				$view_str .= $this->generate_pagination($page,$log_count,ADMINPANEL_LOG_DISPLAY_PER_PAGE,'/adminpanel/logs/feedback/');
				$view_str .= "<table class=\"log\">";
				$view_str .= '<caption class="hidden_context">Log of Feedback Records</caption>';
				$view_str .= "<tr><th>Log Date";
				if(isset($timezone_abbrev)) { $view_str .= ' (' . $timezone_abbrev .')'; }
				$view_str .= "</th><th>Username</th><th>Feedback Type</th><th>Feedback Comments</th></tr>";
				for($i = 0; $i < $feedback->num_rows(); $i++) {
					$row = $feedback->row_array($i);
					$actor_id_query = $this->db->query("SELECT user_name FROM users WHERE user_id=" . $this->db->escape($row["user_id"]));
					if($actor_id_query) {
						$actor_id_row = $actor_id_query->row_array();
						$actor_name = $actor_id_row ["user_name"];
						$view_str .= "<tr><td>" . htmlentities(date("m/d/y h:i:s A",$row["feedback_datetime"])) . "</td><td>" . htmlentities($actor_name) . "</td><td>" . htmlentities($row["feedback_type"]) . "</td><td>" . htmlentities($row["feedback_comments"]) . "</td></tr>";
					}
				}
				$view_str .= "</table>";
			}
		}
		else { show_404(); }
		$data["log_view"] = $view_str;
		return $data;
	}	
	
	/* This function takes inputs and uses them to create a table which can then be transformed by the jQuery visualization
	 * library.
	 */
	protected function markup_for_jquery_chart($chartType, $caption, $data, $id, $width="325px", $height="100px", $colors="['red']", $visualize=TRUE) {
		$chart_table = "<table class=\"chart\" id=\"" . $id . "\" style=\"width: " . $width ."; height: " . $height . ";\">\n";
		$chart_table .= "<caption>" . $caption . "</caption>\n";
		$chart_table .= "<tbody>\n";
		if($chartType != "table") { $chart_table .= "<thead>\n"; }
		$i = 0;
		foreach($data as $row) {
			if($i == 0 && $chartType != "table") { $chart_table .= "<tr>\n"; }
			$j = 0;
			foreach($row as $col) {
				if($j == 0 && $chartType != "table") { $chart_table .= "<th scope=\"col\">" .  $col . "</th>\n"; }
				$j++;
			}
			if($i+1 == count($data) && $chartType != "table") { $chart_table .= "</tr>\n"; }
			$i++;
		}
		if($chartType != "table") { $chart_table .= "</thead>\n"; }
		$i = 0;
		foreach($data as $row) {
			if($i == 0 || $chartType == "pie" || $chartType == "table") { $chart_table .= "<tr>\n"; }
			$j = 0;
			foreach($row as $col) {
				if($j != 0) { $chart_table .= "<td>" .  $col . "</td>\n"; }
				else if(($chartType == "pie" || $chartType == "table") && $j == 0) { $chart_table .= "<th scope=\"row\">" .  $col . "</th>\n";}
				$j++;
			}
			if($i+1 == count($data) || $chartType == "pie" || $chartType == "table") { $chart_table .= "</tr>\n"; }
			$i++;
		}
		$chart_table .= "</tbody>\n";
		$chart_table .= "</table>\n";
		if($visualize == TRUE) {
			$chart_table .= "<script>
			$(document).ready(
				function() { $('#" . $id . "').css('display','none');
				$('#" . $id . "').visualize({type: '" .$chartType . "', width: '" . $width . "', height: '" .$height. "', colors: ". $colors ."}); }
			);
			</script>\n";
		}
		return $chart_table; 
	}
	
	protected function generate_pagination($page,$items,$items_per_page,$link) {
		$view_str = '<h3 class="hidden_context">Pagination</h3>';
		if($items > $items_per_page) {
			$view_str .= "<div id=\"pagination\">\n";
			if($items%$items_per_page > 0) { $page_count = (int)($items/$items_per_page)+1; }
			else { $page_count = (int)($items/$items_per_page); }
			//style pagination
			$before_span = 5;
			$after_span = 5;
			if($page - 5 <= 0) { $after_span += abs($page - $before_span); }
			else if($page + 5 > $page_count) { $before_span += abs($page_count - ($page + $after_span)); }
			for($i = ($page - $before_span); $i <= ($page+$after_span); $i++) {
				if($i > 0 && $i <= $page_count) {
					if($i == ($page - $before_span) && ($page - $before_span > 1)) { $view_str .= '<a style="margin-right: 3px;" href="' . $link . ($page - 1). '"><img src="/images/lt.png" alt="Previous Page"></a>'; }
					if(($i) != $page) { $view_str .= '<a href="' . $link . ($i) . '">' . ($i) . '</a> '; }
					else {  $view_str .= "<span style=\"padding: 2px; font-weight: bold;\">" . ($i) . '</span>'; }
					if($i == ($page+$after_span) && ($page+$after_span < $page_count)) { $view_str .= '<a href="' . $link . ($page + 1). '"><img src="/images/gt.png" alt="Next Page"></a>'; }
				}
			}
			$view_str .= "</div>\n";
		}
		return $view_str;
	}
	
	protected function sanitize_export($input){
		return str_replace("\"","\"\"",str_replace(array("\r","\n"),"",$input));
	}
		
	public function valid_formatted_address($str)
	{
		$valid_format = (!preg_match("/^([a-z0-9\+_\-]+)(\.[a-z0-9\+_\-]+)*@([a-z0-9\-]+\.)+[a-z]{2,6}$/ix", $str)) ? FALSE : TRUE;
		return $valid_format;
	}
		

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// USER HELPER FUNCTIONS
// These functions are not web-accessible and at some point we should evaluate whether or not they would more appropriately belong in models/libraries.
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////		
	
	protected function can_edit_user($username){
		if($this->has_permission("manage_users_all")){
			return true;
		}else if($this->has_permission("manage_users_facilities")){
			$resource = '/direct/account/in_facility/mailbox/'.$this->session->userdata('username');
			$resource .= '/format/json';
			$response = @$this->api_model->webservice_call($resource,'GET');
			if($response->http_status === 200){
				$users = $response->response->accounts;
				if( in_array($username, $users)){
					return true;
				}
			}
		}
		show_404();
	}
	
	protected function create_user() {
		$this->verify_permission("manage_users_create");
		$this->load->model('webmailmodel');
		$this->form_validation->set_rules("first_name","First Name","required","xss_clean");
		$this->form_validation->set_rules("last_name","Last Name","required","xss_clean");
		$this->form_validation->set_rules("username","Username","required|callback_allowed_mailbox_characters","xss_clean");
		$this->form_validation->set_rules("ext_mail","External Email","callback_va_email","xss_clean");
	
		$edipi_post = $this->input->post('edipi',TRUE);
		$piv_post = $this->input->post('piv',TRUE);
			
		$this->form_validation->set_message('is_natural', 'The %s field must contain only numeric characters.');
		if(!isset($piv_post) || strlen(trim($piv_post)) <= 0) { $this->form_validation->set_rules('edipi','EDIPI','callback_id_required[edipi]|is_natural'); }
		else { $this->form_validation->set_rules('edipi','EDIPI','is_natural'); }
		if(!isset($edipi_post)  || strlen(trim($edipi_post)) <= 0) { $this->form_validation->set_rules('piv','PIV ID','callback_id_required[piv]|is_natural'); }
		else { $this->form_validation->set_rules('piv','PIV ID','is_natural'); }
		
		if($this->form_validation->run() == true) {
			//gather post items into variables
			$username_post = $this->input->post("username",TRUE);
			$first_post = $this->input->post("first_name",TRUE);
			$last_post = $this->input->post("last_name",TRUE);
			$ext_mail_post = $this->input->post("ext_mail",TRUE);
			$middle_post = $this->input->post("middle_name",TRUE);
			$organization_post = $this->input->post("organization",TRUE);
			$department_post = $this->input->post("department",TRUE);
			$telephone_post = $this->input->post("telephone",TRUE);
			$mobile_post = $this->input->post("mobile",TRUE);
			$location_post = $this->input->post("location",TRUE);
			$title_post = $this->input->post("title",TRUE);
			//process required items
			//do piv/edipi id number first
			if((isset($edipi_post) && strlen($edipi_post) > 0) && (isset($piv_post) && strlen($piv_post) > 0)) { $edipi = $edipi_post;  $piv_id = $piv_post; } 
			else if(isset($edipi_post) && strlen($edipi_post) > 0) { $edipi = $edipi_post; }
			else if(isset($piv_post) && strlen($piv_post) > 0) { $piv_id = $piv_post; }
			else { return FALSE; }
			//then do other required items
			if(isset($username_post) && strlen($username_post) > 0) { $username = $username_post; } else { return FALSE; }
			if(isset($first_post) && strlen($first_post) > 0) { $first = $first_post; } else { return FALSE; }
			if(isset($last_post) && strlen($last_post) > 0) { $last = $last_post; } else { return FALSE; }
			//process optional items
			if(isset($ext_mail_post) && strlen($ext_mail_post) > 0) { $ext_mail = $ext_mail_post; } else { $ext_mail = ""; } //this is ext. mail and could be blank in rare cases, so we make it optional and can be manually changed if needed
			if(isset($middle_post) && strlen($middle_post) > 0) { $middle = $middle_post; } else { $middle = NULL;  }
			if(isset($organization_post) && strlen($organization_post) > 0) { $organization = $organization_post; } else { $organization = NULL;  }
			if(isset($department_post) && strlen($department_post) > 0) { $department = $department_post; } else { $department = NULL;  }
			if(isset($telephone_post) && strlen($telephone_post) > 0) { $telephone = $telephone_post; } else { $telephone = NULL;  }
			if(isset($title_post) && strlen($title_post) > 0) { $title = $title_post; } else { $title = NULL;  }
			if(isset($mobile_post) && strlen($mobile_post) > 0) { $mobile = $mobile_post; } else { $mobile = NULL;  }
			if(isset($location_post) && strlen($location_post) > 0) { $location = $location_post; } else { $location = NULL;  }
			//create random password
			$pass = User::random_password();
			if(isset($pass) && strlen($pass) > 0) { $userpassword = $this->encrypt->ssha256_encode($pass); } else { return FALSE; }
			if(!isset($userpassword) || !(strlen($userpassword) > 0)) { return FALSE; }
			
			if(!empty($username_post) && User::exists(array('user_name' => $username_post))) { 
				$this->session->set_error_message('Username already exists. User creation failed.');
				return FALSE; 
			}
			if(!empty($edipi_post) && User::exists(array('user_edipi' => $edipi_post))) { 
				$this->session->set_error_message('EDIPI already exists. User creation failed.');
				return FALSE; 
			}
			if(!empty($piv_post) && User::exists(array('user_piv_id' => $piv_post))) { 
				$this->session->set_error_message('PIV ID already exists. User creation failed.');
				return FALSE; 
			}
			
			if($this->ldap->connected()) {
				$attributes = array();
				$attributes["objectClass"] = array("posixAccount", "top", "person", "organizationalPerson", "inetOrgPerson");
				$attributes["gidNumber"] = "5000";
				$attributes["uidNumber"] = "5000";
				$attributes["uid"] = $username;
				$attributes["homeDirectory"] = "/var/mailboxes/" . $username;
				$attributes["cn"] =($first . " " . $last);
				$attributes["givenName"] = $first;
				$attributes["sn"] = $last;
				$attributes["mail"] = $username . "@" . DIRECT_DOMAIN;
				$attributes["userPassword"] = $userpassword;
				if(!is_null($middle)) { $attributes["initials"] = $middle; $attributes["displayName"] = ($last . ", " . $first . " " . $middle); }
				else { $attributes["displayName"] = ($last . ", " . $first); }
				if(!is_null($department)) { $attributes["departmentNumber"] = $department; }
				if(!is_null($organization)) { $attributes["o"] = $organization; }
				if(!is_null($telephone)) { $attributes["telephoneNumber"] = $telephone; }
				if(!is_null($mobile)) { $attributes["mobile"] = $mobile; }
				if(!is_null($location)) { $attributes["physicaldeliveryofficename"] = $location; }
				if(!is_null($title)) { $attributes["title"] = $title; }
			
				if(!$this->ldap->create_ldap_account($attributes)) { 
					$this->session->set_error_message('Failed to create LDAP account. User creation failed.');
					return FALSE; 
				}
			}
			else { return FALSE; }
			$get_actor_id = $this->webmailmodel->get_user_id($this->session->userdata('username'));
			if($get_actor_id && $get_actor_id->num_rows() == 1) {
				$row = $get_actor_id->row_array();
				$actor_id = $row["user_id"];
				$db_attributes = array(
					'edipi' => (isset($edipi)) ? $edipi : '',
					'piv_id' => $piv_id,
					'username' => $username,
					'ext_mail' => $ext_mail,
					'pass' => $pass,
					'actor_id' => $actor_id,
					'user_ext_notify_flag' => 1,
					'user_ext_group_notify_flag' => 1,
					'user_deleted_flag' => 0,
					'user_is_group' => 0,
				);
				$create_user_rec = $this->webmailmodel->create_user_record($db_attributes);
				if($create_user_rec == FALSE) { 
					$this->session->set_error_message('Failed to enter user into database. User creation failed.');
					return FALSE; 
				}
				
				
				//if we get to here it should have succeeded, so we log it
				$get_actor_id = $this->webmailmodel->get_user_id($this->session->userdata('username'));
				$get_target_id = $this->webmailmodel->get_user_id($username);
				if($get_actor_id && $get_target_id) {
					$actor_id_row = $get_actor_id->row_array();
					$target_id_row = $get_target_id->row_array();
					$actor_id = $actor_id_row["user_id"];
					$target_id = $target_id_row["user_id"];
					$this->load->library("audit");
					$this->audit->log_event("edit",array($target_id,$actor_id,"Create User",date('U')));
				}
				
				$this->session->set_success_message('Account successfully created.');
				redirect('adminpanel/manage_users/');
			}
			else { 
				$this->session->set_error_message('Failed to enter user into database. User creation failed.');
				return FALSE; }
		}
		else {
			$form_data = $this->input->post(NULL,TRUE);
			$validation_errors = array();
			if(is_array($form_data)) {
				foreach($form_data as $key => $input) {
					if(strlen(form_error($key)) > 0) { $validation_errors[$key] = form_error($key); }
				}
				$this->session->set_flashdata('validation_errors',$validation_errors);
			}
			return FALSE;
		}
	}
	
	/* This searches the address book and returns a list of users for use with the search in edit distribution list view */
	protected function get_user_search($input = null){  
        $input = urldecode($input);
		$result_arr = array();
		//get addresses from global address book in LDAP
		$ldapconfig['user'] = $this->session->userdata('username');
		$ldapconfig['pwd'] = $this->encrypt->decode($this->session->userdata('ep'));
		$this->load->library('ldap',$ldapconfig);
		$entries = ($this->ldap->search($input));
        $i = 0;
		foreach($entries as $key => $val) {
			if($this->valid_formatted_address($val['mail'])) {
				$contact_arr = array('name'=>(isset($val['displayname'])?$val['displayname']:"Undefined"),'id'=>$val['mail']);
				array_push($result_arr,$contact_arr);
				$i++;
			}
		}
		
		//sort combined result array (natural order, case insensitive)
		usort($result_arr, function( $el1, $el2) { return strnatcasecmp( $el1['name'], $el2['name']); });
		
		//add whatever the user is currently typing so that it is allowed as well
		if($this->valid_formatted_address($input)) {
			$contact_arr = array('name' => $input, 'id' => $input);
			array_push($result_arr,$contact_arr);
		}
		array_walk_recursive($result_arr, function (&$value) {
			$value = htmlentities($value);
		});
		return $result_arr;
	}		
	
	protected function remove_user() {
		$uid = $this->input->post("uid",TRUE);
		$this->can_edit_user($uid);
		//check that uid is valid
		if(is_null($uid) || strlen($uid) <= 0) { show_404(); }
		$get_user = $this->db->query("SELECT user_name FROM users WHERE user_deleted_flag=0 AND user_name=" . $this->db->escape($uid));
		if($get_user && $get_user->num_rows() == 1) { $uid_valid = TRUE; } else { show_404(); }
		//check if its valid, but equal to the current user
		if($uid_valid && $uid == $this->session->userdata("username")) {
			$this->session->set_error_message('Cannot remove logged in user account.');			
			redirect("adminpanel/manage_users/user/" . $uid); //don't let them delete themselves
		}
		//if valid
		if($uid_valid) {
			if($this->ldap->delete_ldap_account($uid)) {
				$update_flag = $this->db->query("UPDATE users SET user_deleted_flag=1 WHERE user_name=" . $this->db->escape($uid));
				if($update_flag) { 
					$get_actor_id = $this->db->query("SELECT user_id FROM users WHERE user_deleted_flag = 0 AND user_name=" . $this->db->escape($this->session->userdata("username")));
					$get_target_id = $this->db->query("SELECT user_id FROM users WHERE user_deleted_flag = 1 AND user_name=" . $this->db->escape($uid));
					if($get_actor_id && $get_target_id) {
						$actor_id_row = $get_actor_id->row_array();
						$target_id_row = $get_target_id->row_array();
						$actor_id = $actor_id_row["user_id"];
						$target_id = $target_id_row["user_id"];
						$this->load->library("audit");
						$this->audit->log_event("edit",array($target_id,$actor_id,"Remove User",date('U')));
					}
					$this->session->set_success_message('Removed account successfully.');					
					redirect("/adminpanel/manage_users/"); 
				}
				else { 
					$this->session->set_error_message('Removed from LDAP, however, failed to remove account in database. Contact a system administrator for assistance.');					
					redirect("/adminpanel/manage_users/user/" . $uid); 
				}
			}
			else {
				$this->session->set_error_message('Failed to delete account.');	
				redirect("/adminpanel/manage_users/user/" . $uid);
			}
		}
	}
	
	protected function restore_user() {
		$uid = $this->input->post("uid",TRUE);
		$this->can_edit_user($uid);
		//check that uid is valid
		if(is_null($uid) || strlen($uid) <= 0) { show_404(); }
		$get_user = $this->db->query("SELECT user_name FROM users WHERE user_deleted_flag=1 AND user_name=" . $this->db->escape($uid));
		if($get_user && $get_user->num_rows() == 1) { $uid_valid = TRUE; } else { show_404(); }
		
		//if valid
		if($uid_valid) {
			if($this->ldap->restore_ldap_account($uid)) {
				$update_flag = $this->db->query("UPDATE users SET user_deleted_flag=0 WHERE user_name=" . $this->db->escape($uid));
				if($update_flag) { 
					$get_actor_id = $this->db->query("SELECT user_id FROM users WHERE user_deleted_flag = 0 AND user_name=" . $this->db->escape($this->session->userdata("username")));
					$get_target_id = $this->db->query("SELECT user_id FROM users WHERE user_deleted_flag = 0 AND user_name=" . $this->db->escape($uid));
					if($get_actor_id && $get_target_id) {
						$actor_id_row = $get_actor_id->row_array();
						$target_id_row = $get_target_id->row_array();
						$actor_id = $actor_id_row["user_id"];
						$target_id = $target_id_row["user_id"];
						$this->load->library("audit");
						$this->audit->log_event("edit",array($target_id,$actor_id,"Remove User",date('U')));
					}
					$this->session->set_success_message('Restored account successfully.');					
					redirect("/adminpanel/manage_users/"); 
					
				}
				else { 
					$this->session->set_error_message('Restored in LDAP, however, failed to restore account in database. Contact a system administrator for assistance.'); 					
					redirect("/adminpanel/manage_users/user/" . $uid); }
			}
			else {
				$this->session->set_error_message('Failed to restore account.'); 
				redirect("/adminpanel/manage_users/user/" . $uid);
			}
		}
	}		

	protected function update_user_account($active_account = TRUE) {
		$management = ($this->input->post("management",TRUE) === "TRUE");
		if($management){
			$this->form_validation->set_rules("first_name","First Name","required","xss_clean");
			$this->form_validation->set_rules("last_name","Last Name","required","xss_clean");
			$this->form_validation->set_rules('facility_id','Facility','required');
		}
		if($this->input->post("confirm_user_pwd",TRUE) == "on")  {
			$this->form_validation->set_rules("user_pwd","User Password","required|matches[reenter_user_pwd]|callback_check_password_strength");
			$this->form_validation->set_rules("reenter_user_pwd","Re-Enter User Password","required|matches[user_pwd]");
		}

		$uid = $this->input->post("uid",TRUE);
		$this->can_edit_user($uid);
		$this->form_validation->set_rules("uid","required");
		//check that uid is valid
		if(is_null($uid) || strlen($uid) <= 0) { show_404(); }
		$get_user = $this->db->query("SELECT user_name FROM users WHERE user_name=" . $this->db->escape($uid));
		if($get_user && $get_user->num_rows() == 1) { $uid_valid = TRUE; } else { show_404(); }
		if($this->form_validation->run() == true && $uid_valid) {
			//capture input for ldap
			$first = $this->input->post("first_name",TRUE);
			$last = $this->input->post("last_name",TRUE);
			$middle = $this->input->post("middle_name",TRUE);
			$title = $this->input->post("job_title",TRUE);
			$department = $this->input->post("department",TRUE);
			$organization = $this->input->post("organization",TRUE);
			$telephone = $this->input->post("telephone",TRUE);
			$mobile = $this->input->post("mobile",TRUE);
			$location = $this->input->post("location",TRUE);
			$admin_level = $this->input->post("admin_level",TRUE);
			if($this->input->post("confirm_user_pwd",TRUE) == "on")  {
				$user_pwd = $this->input->post("user_pwd",TRUE);
			}
			
			$attributes = array();
			//add mandatory attrs to array
			$attributes["givenName"] = $first;
			$attributes["sn"] = $last;
			$attributes["cn"] = $first . " " . $last;
		
			//deal with optional attributes (middle name is a sepcial case, since we need to change displayName based on it too)
			if(isset($middle) && strlen($middle) > 0) { $attributes["initials"] = $middle; $attributes["displayName"] = $last . ", " . $first . " " . $middle; }
			else { $attributes["displayName"] = $last . ", " . $first; }
			if(isset($title) && strlen($title) > 0) { $attributes["title"] = $title; }
			if(isset($department) && strlen($department) > 0) { $attributes["departmentnumber"] = $department; }
			if(isset($organization) && strlen($organization) > 0) { $attributes["o"] = $organization; }
			if(isset($telephone) && strlen($telephone) > 0) { $attributes["telephonenumber"] = $telephone; }
			if(isset($mobile) && strlen($mobile) > 0) { $attributes["mobile"] = $mobile; }
			if(isset($location) && strlen($location) > 0) { $attributes["physicaldeliveryofficename"] = $location; }
            if(!$management){
            	$attributes = array();
            }
			if(isset($user_pwd) && strlen($user_pwd) > 0) { $attributes["userPassword"] = $this->encrypt->ssha256_encode($user_pwd); }
			
			//check show/hide personal mailbox
			if(array_key_exists('show_personal',$this->input->post(NULL,TRUE))) { $attributes['employeetype'] = ''; } 
			else { $attributes['employeetype'] = 'mailboxhidden'; }
			
			//check inactive vs. active account for correct dn
            if($active_account) { $dn = LDAP_ACCOUNTS_DN; }
            else { $dn = LDAP_DELETED_ACCOUNTS_DN; }
            
            $resource = '/direct/account/update/format/json';
            $fields = $this->input->post();
            //construct post request
            $fields ['mailbox'] = $uid;
            $post = '';
            foreach($fields as $key => $value) { $post .= $key.'='.urlencode($value).'&'; } //url-ify the data for the POST
            $post = rtrim($post, '&');
            if($management){
            	$response = $this->api_model->webservice_call($resource,'POST',$post);
            }
            if(!$management || $response->http_status === 200){
				//modify ldap account with new attributes
				$modified = $this->ldap->modify_ldap_account($uid,$attributes,$dn);
				
				//set admin level
				$group_search = $this->ldap->get_admin_group_membership("uid=" . $uid . "," . $dn);
				$groups = array();
				for($i = 0; $i < $group_search["count"]; $i++) {
					array_push($groups,$group_search[$i]["dn"]);
				}

				if($this->has_permission('admin')){
					if(in_array(LDAP_ADMIN_GROUP1,$groups)) { $prev_admin_level = 1; }
					else { $prev_admin_level = 0; }
					
					if($prev_admin_level != $admin_level) {
						$admin_level_modify = $this->ldap->set_admin_group_membership($uid,$admin_level);
					}
					else { $admin_not_changed = TRUE; }
				}
				//capture input for database
				$ext_mail = $this->input->post("ext_mail",TRUE);
				$notify = array_key_exists("notify",$this->input->post(NULL,TRUE));
				$group_notify = array_key_exists('group_notify',$this->input->post(NULL,TRUE));
				//don't update database unless ldap succeeds
				
				//roles
				if($this->has_permission('admin')){
					$roles = array_intersect_key($this->input->post(null,TRUE), array_flip(preg_grep('/^role_/', array_keys($this->input->post(null,TRUE)))));
					$current_roles = $this->role_model->get_roles_membership("uid=" . $uid . "," . $dn);
					$roles_add = array_diff($roles, $current_roles);
					$roles_remove = array_diff($current_roles,$roles);
					$this->role_model->remove_roles_from_member($roles_remove,"uid=" . $uid . "," . $dn);
					$this->role_model->add_roles_to_member($roles_add,"uid=" . $uid . "," . $dn);
				}	
				
				
				if($modified) {
					if(isset($user_pwd) && strlen($user_pwd) > 0) {
						$update = $this->db->query("UPDATE users SET user_ep=".$this->db->escape($this->encrypt->encode($user_pwd)).",user_mail=" . $this->db->escape($ext_mail).", user_ext_notify_flag=".$this->db->escape($notify).", user_ext_group_notify_flag=".$this->db->escape($group_notify)." WHERE user_name=" . $this->db->escape($uid));
						if($uid == $this->session->userdata('username')) { 
							if($update) { 
								$this->session->set_userdata('ep',$this->encrypt->encode($user_pwd)); //update user pwd in session if its the current user
							}
						}
					}
					else {
						$update = $this->db->query("UPDATE users SET user_mail=".$this->db->escape($ext_mail).", user_ext_notify_flag=".$this->db->escape($notify).", user_ext_group_notify_flag=".$this->db->escape($group_notify)." WHERE user_name=" . $this->db->escape($uid));
					}
					//remove personal notifications to external email if personal mailbox access is turned off
					if($attributes['employeetype'] == 'mailboxhidden') { 
						$update_notifications = $this->db->query("UPDATE users SET user_ext_notify_flag=0 WHERE user_name=".$this->db->escape($uid));
						//if the current user is having their mailbox hidden, update the current session to reflect that
						if($uid == $this->session->userdata('username')) { $this->session->set_userdata('hide_personal_mailbox',TRUE); }
					}
					else {
						//if the current user is having their mailbox unhidden, update the current session to reflect that
						if($uid == $this->session->userdata('username')) { $this->session->set_userdata('hide_personal_mailbox',FALSE); }
					}
				}
				if($modified && $update) {
					$this->load->library('audit');
					
					$message = 'Updated user settings.';
					$this->session->set_success_message($message);
					
					//update cn for logged in user if it is the one being updated
					if($this->session->userdata('username') == $uid) { 
						$this->session->set_userdata('user_cn',$attributes['cn']); //update user's name in session
					}
					//log update for auditing
					$actor_id = $this->user->id;
					$target_id = $this->get_user_id($uid);
					$this->audit->log_event('edit',array($target_id,$actor_id,'Update user information',date('U')));
					if(!$admin_not_changed && $this->has_permission('admin')) {
						if($admin_level_modify && ($admin_level == 1) && ($prev_admin_level == 0)) {
							$this->session->set_success_message($message.' '.'Added user as an administrator.');							
							$this->audit->log_event("edit",array($target_id,$actor_id,"Added user to administrators.",date('U')));
						}
						else if($admin_level_modify && ($admin_level == 0) && ($prev_admin_level == 1)) {
							$this->session->set_success_message($message.' '.'Removed user as an administrator.');							
							$this->audit->log_event("edit",array($target_id,$actor_id,'Removed user from administrators.',date('U')));
						}
						else if(!$admin_level_modify) {
							$this->session->set_error_message('Failed to update user administrative level.');
						}
					}
					redirect('/adminpanel/manage_users/user/' . $uid);
				}
				else { 
					$this->session->set_error_message('Failed to update user.');
					redirect('/adminpanel/manage_users/user/' . $uid);
				}
			}
			else{
				$this->session->set_error_message('Failed to update direct account.');
				redirect('/adminpanel/manage_users/user/' . $uid);
			}
		}
		else { 
			$form_data = $this->input->post(NULL,TRUE);
			$validation_errors = array();
			if(is_array($form_data)) {
				foreach($form_data as $key => $input) {
					if(strlen(form_error($key)) > 0) { $validation_errors[$key] = form_error($key); }
				}
				$this->session->set_flashdata('validation_errors',$validation_errors);
			}
			redirect('/adminpanel/manage_users/user/' . $uid); 
		}
	}	
	
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// FORM VALIDATION CALLBACKS
// These are currently public - iddeally, these should be protected or just added to the form validation library so that they can be used throughout the application
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	public function allowed_mailbox_characters($name) {
		$valid = (preg_match('/[^A-Za-z0-9._-]/', $name) === 1) ? FALSE : TRUE;
		if($valid) { return TRUE; }
		$this->form_validation->set_message('allowed_mailbox_characters', 'The %s field may only contain alpha-numeric characters (A-Z, a-z, 0-9), underscores, periods, and dashes.');
		return FALSE;
	}

	public function check_password_strength($password) {
		$valid = TRUE;
		$msg = 'Password strength requirements not met.<br />';
		if(strlen($password) < 15) {
			$msg .= 'Must be at least 15 characters long.<br />';
			$valid = FALSE;
		}
		if(!preg_match("/[a-z]+/", $password)) {
			$msg .= 'Must contain at least one lowercase character.<br />';
			$valid = FALSE;
		}
		if(!preg_match("/[A-Z]+/", $password)) {
			$msg .= 'Must contain at least one uppercase character.<br />';
			$valid = FALSE;
		}
		if(!preg_match("/[0-9]+/", $password)) {
			$msg .= 'Must contain at least one numeric character.<br />';
			$valid = FALSE;
		}
		if(!preg_match("/[!@#$%&*()_+|~=`{}[:;'<>?,.\-\^\]\\\\]+/", $password)) { //four backslashes to escape the backslash once for PHP and once for the regex
			$msg .= 'Must contain at least one special character.<br />';
			$valid = FALSE;
		}
		if(!$valid) { $this->form_validation->set_message('check_password_strength', $msg); }
		return $valid;
	}

	/* Form validation callback */
	public function list_is_unique($name) {
		//if we're editing, we could be using the public list model or the deleted list model.  otherwise, it will 
		if(!isset($this->list_model))
			$this->load->model('public_distribution_list_model', 'list_model');
		
		$this->form_validation->set_message('list_is_unique', 'The %s field must not match any existing lists or removed lists.');
		
		$name_is_available = $this->list_model->name_is_available($name);
		if($name_is_available) return true;
			
		//if we're editing (edit_distribution_list or edit_removed_distribution_list), check to see if the name is being used by the list we're editing
		if(string_begins_with('edit', $this->router->method) && string_ends_with('distribution_list', $this->router->method)){
			$list = $this->list_model->find_one(compact('name'));
			return $list['id'] == $this->input->post('list_id',TRUE);			
		}
		
		return $name_is_available;
	}

	public function va_email($email) {
		$valid = (preg_match("/^([a-z0-9\+_\-]+)(\.[a-z0-9\+_\-]+)*@+((.*(\.va\.gov))|va\.gov)$/ix", $email)) ? TRUE : FALSE;
		
		if($valid) { return TRUE; }
		$this->form_validation->set_message('va_email', 'The %s field must contain a valid domain email address.');
		return FALSE;
	}	
	
	public function valid_trusted_address($str) {
		//have to urlencode / base64 encode the address to send to the web service
		$str = rawurlencode(base64_encode($str));
		$resource = '/direct/validate/address/'.$str.'/format/json';
		$url = WEBSERVICE_URL . $resource;
		
		$headers = array(
			'Authorization: DPII ' . WEBSERVICE_PUBLIC_KEY . ':'. base64_encode(hash_hmac('sha256',"GET\n" . date('U'). "\n" . $resource, WEBSERVICE_PRIVATE_KEY)),
			'Date: ' . date('U'),
			);
		$ch = curl_init();
		curl_setopt($ch,CURLOPT_URL, $url);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
		curl_setopt($ch, CURLOPT_HTTPHEADER,$headers);
		$server_output = curl_exec($ch);
		if($server_output === false)
			trigger_error( curl_error($ch), E_USER_WARNING);
		$http_status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
		$response = $this->json->decode($server_output);
		if(isset($response->valid)) {
			$valid = $response->valid;
			if(!$valid) { $this->form_validation->set_message('valid_trusted_address', 'The %s field must be a trusted Direct Address'); }
			return $valid;
		}
		else { $this->form_validation->set_message('valid_trusted_address', 'The %s field must be a trusted Direct Address'); return FALSE; }
	}
		
	
}
/* End of file adminpanel.php */
/* Location: ./application/controllers/adminpanel.php */
